﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Media;
using System.Windows.Input;

namespace DrawingInWPF
{
    // Create a host visual derived from the FrameworkElement class.
    // This class provides layout, event handling, and container support for
    // the child visual objects.
    public class MyVisualHost : FrameworkElement
    {
        // Create a collection of child visual objects.
        private VisualCollection m_VisualChildren;

        private ZoomPan m_ZoomPan = new ZoomPan(); // Zooming and panning controler
        private Point m_LastMousePoint; // Last point
        private bool m_Panning = false; // Is panning active (left mouse button clicked)

        public MyVisualHost()
        {
            // Initialize visual children collection
            m_VisualChildren = new VisualCollection(this);

            // Create pen
            Pen MyPen1 = new Pen(Brushes.Red, 3);
            Pen MyPen2 = new Pen(Brushes.White, 3);

            Point pt1 = new Point(50, -100);
            Point pt2 = new Point(170, -50);
            Point pt3 = new Point(400, -200);
            Point pt4 = new Point(200, -10);
            Point pt5 = new Point(50, -100);

            List<Point> otherPoints = new List<Point>(4);
            otherPoints.Add(pt2);
            otherPoints.Add(pt3);
            otherPoints.Add(pt4);
            otherPoints.Add(pt5);

            m_ZoomPan.Width = 500;
            m_ZoomPan.Height = 300;

            // Create background panel
            DrawingVisual background = CreateDrawingVisualRectangle(MyPen2, new Point(0, 0), 
                new Point(System.Windows.SystemParameters.WorkArea.Width, 
                    System.Windows.SystemParameters.WorkArea.Height));
            m_VisualChildren.Add(background);

            // Create and add two ellispes to visual collection
            DrawingVisual polygon = CreateDrawingVisualPolygon(MyPen1, pt1, otherPoints, m_ZoomPan.Transform);
            m_VisualChildren.Add(polygon);   

            this.MouseLeftButtonDown += new MouseButtonEventHandler(MyVisualHost_MouseLeftButtonDown);
            this.MouseLeftButtonUp += new MouseButtonEventHandler(MyVisualHost_MouseLeftButtonUp);
            this.MouseWheel += new MouseWheelEventHandler(MyVisualHost_MouseWheel);
            this.MouseMove += new MouseEventHandler(MyVisualHost_MouseMove);


            double xMax = 400;
            double xMin = 50;

            double yMax = -10;
            double yMin = -200;

            double x = (xMax + xMin) / 2;
            double y = (yMax + yMin) / 2;

            double xDelta = Math.Abs(xMax - xMin);
            double yDelta = Math.Abs(yMax - yMin);

            double xRatio = 500 / xDelta;
            double yRatio =  300 / yDelta;

            double ratio = (xRatio < yRatio) ? xRatio : yRatio;

            m_ZoomPan.ZoomAll(x, y, ratio);
        }

        private void MyVisualHost_MouseLeftButtonDown(object sender, MouseEventArgs e)
        {
            m_Panning = true;
            m_LastMousePoint = e.GetPosition(this);
        }

        private void MyVisualHost_MouseLeftButtonUp(object sender, MouseEventArgs e)
        {
            m_Panning = false;
        }

        private void MyVisualHost_MouseMove(object sender, MouseEventArgs e)
        {
            Point HitPoint = e.GetPosition(this);

            if (m_Panning)
            {
                m_ZoomPan.Pan(HitPoint.X - m_LastMousePoint.X, HitPoint.Y - m_LastMousePoint.Y);
            }

            m_LastMousePoint = HitPoint;
        }

        private void MyVisualHost_MouseWheel(object sender, MouseWheelEventArgs e)
        {
            if (e.Delta > 0) m_ZoomPan.ZoomIn();
            else m_ZoomPan.ZoomOut();
        }


        // Provide a required override for the VisualChildrenCount property.
        protected override int VisualChildrenCount
        {
            get { return m_VisualChildren.Count; }
        }

        // Provide a required override for the GetVisualChild method.
        protected override Visual GetVisualChild(int index)
        {
            if (index < 0 || index >= m_VisualChildren.Count)
            {
                throw new ArgumentOutOfRangeException();
            }

            return m_VisualChildren[index];
        }

        // Create a DrawingVisual that contains a line.
        private DrawingVisual CreateDrawingVisualLine(Pen pen, Point startPt, Point endPt)
        {
            // Initialize drawing visual
            DrawingVisual drawingVisual = new DrawingVisual();

            // Retrieve the DrawingContext in order to create new drawing content.
            DrawingContext drawingContext = drawingVisual.RenderOpen();

            // Draw line in the DrawingContext.
            LineGeometry lineGeometry = new LineGeometry(startPt, endPt);
            drawingContext.DrawGeometry(null, pen, lineGeometry);

            // Persist the drawing content.
            drawingContext.Close();

            return drawingVisual;
        }

        // Create a DrawingVisual that contains a rectangle.
        private DrawingVisual CreateDrawingVisualRectangle(Pen pen, Point startPt, Point endPt)
        {
            // Initialize drawing visual
            DrawingVisual drawingVisual = new DrawingVisual();

            // Retrieve the DrawingContext in order to create new drawing content.
            DrawingContext drawingContext = drawingVisual.RenderOpen();

            // Draw rectangle in the DrawingContext.
            RectangleGeometry lineGeometry = new RectangleGeometry(new Rect(startPt, endPt));
            drawingContext.DrawGeometry(pen.Brush, pen, lineGeometry);

            // Persist the drawing content.
            drawingContext.Close();

            return drawingVisual;
        }

        // Create a DrawingVisual that contains a ellipse.
        private DrawingVisual CreateDrawingVisualEllipse(Pen pen, Point center, double radiusX, double radiusY)
        {
            // Initialize drawing visual
            DrawingVisual drawingVisual = new DrawingVisual();

            // Retrieve the DrawingContext in order to create new drawing content.
            DrawingContext drawingContext = drawingVisual.RenderOpen();

            // Draw rectangle in the DrawingContext.
            EllipseGeometry ellipseGeometry = new EllipseGeometry(center, radiusX, radiusY);
            drawingContext.DrawGeometry(pen.Brush, pen, ellipseGeometry);

            // Persist the drawing content.
            drawingContext.Close();

            return drawingVisual;
        }

        // Create a DrawingVisual that contains a polygon.
        private DrawingVisual CreateDrawingVisualPolygon(Pen pen, Point startPoint, List<Point> otherPoints, Transform transform)
        {
            // Initialize drawing visual
            DrawingVisual drawingVisual = new DrawingVisual();

            // Retrieve the DrawingContext in order to create new drawing content.
            DrawingContext drawingContext = drawingVisual.RenderOpen();

            // Draw polygon in the DrawingContext.
            StreamGeometry streamGeometry = new StreamGeometry();
            StreamGeometryContext streamGeometryContex = streamGeometry.Open();

            streamGeometryContex.BeginFigure(startPoint, true, true);

            foreach (Point pt in otherPoints)
            {
                streamGeometryContex.LineTo(pt, true, false);
            }

            streamGeometryContex.Close();

            streamGeometry.Transform = transform;

            drawingContext.DrawGeometry(null, pen, streamGeometry);

            // Persist the drawing content.
            drawingContext.Close();

            return drawingVisual;
        }

    }
}
